--=========================================================================================
-- BUSHEL PLUS SCRIPT
--
-- @For:        bcbuhler
-- @Author:     Akuenzi
-- @Date:        2021-04-11 FS19 created by AKuenzi
--              2023-01-18 FS22 converted by Jos
--              2024-11-30 FS25 revised by Jos
-- @Purpose:    Script allows yield improvements using Bushel Plus combine attachment.
-- @Author:     Akuenzi
--=========================================================================================

--[[NOTE:    The 'scale' to use in the vehicle's xml file is the amount of increase/decrease
            one wishes to see.  If one wishes to see a 10% increase in yield as a result of
            attaching this mod, then the scale factor should be set to 0.10.]]

BushelPlus = {}

function BushelPlus.prerequisitesPresent(...)
    return true
end

function BushelPlus.initSpecialization()
    local schema = Vehicle.xmlSchema
    schema:setXMLSpecializationType("BushelPlus")
    schema:register(XMLValueType.FLOAT, "vehicle.bushelPlus#scale", "Threshing scale", 0.1)
    schema:setXMLSpecializationType()
end

function BushelPlus.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", BushelPlus)
    SpecializationUtil.registerEventListener(vehicleType, "onPostAttach", BushelPlus)
    SpecializationUtil.registerEventListener(vehicleType, "onPreDetach", BushelPlus)
end

function BushelPlus:onLoad(...)
    self.spec_BushelPlus = {}
    local spec = self.spec_BushelPlus
    spec.bushelYieldScale = Utils.getNoNil(self.xmlFile:getValue("vehicle.bushelPlus#scale"), 0.1)
end

function BushelPlus:onPostAttach(...)
    local attacherVehicle = self:getAttacherVehicle()

    if attacherVehicle ~= nil and attacherVehicle.verifyCombine ~= nil then
        -- If we're attached to a combine apply Bushel Plus yield adjustment.
        local spec = self.spec_BushelPlus
        local comSpec = attacherVehicle.spec_combine
        comSpec.threshingScale = comSpec.threshingScale * (1 + spec.bushelYieldScale)
    end
end

function BushelPlus:onPreDetach(...)
    local attacherVehicle = self:getAttacherVehicle()

    if attacherVehicle ~= nil and attacherVehicle.verifyCombine ~= nil then
        -- If we're detaching from a combine, re-set the threshing scale.
        local spec = self.spec_BushelPlus
        local comSpec = attacherVehicle.spec_combine
        comSpec.threshingScale = comSpec.threshingScale / (1 + spec.bushelYieldScale)
    end
end