-- =========================================================
-- FS25 Farm Tablet Mod (version 1.1.0.0)
-- =========================================================
-- Digging App - Shows terrain deformation and digging information
-- =========================================================
-- Author: TisonK
-- =========================================================

function FarmTabletUI:loadDiggingApp()
    local content = self.ui.appContentArea
    if not content then
        self:log("No content area in digging app")
        return
    end

    local padX = self:px(15)
    local padY = self:py(15)
    local titleY = content.y + content.height - padY - 0.03

    -- Title
    table.insert(self.ui.appTexts, {
        text = "Digging Information",
        x = content.x + padX,
        y = titleY,
        size = 0.022,
        align = RenderText.ALIGN_LEFT,
        color = self.UI_CONSTANTS.TEXT_COLOR
    })

    local diggingInfo = self:getDiggingInfo()
    local yPos = titleY - 0.035

    -- Check if terrain system is available
    if not diggingInfo.hasTerrainSystem then
        table.insert(self.ui.appTexts, {
            text = "No terrain deformation system detected",
            x = content.x + padX,
            y = yPos,
            size = 0.016,
            align = RenderText.ALIGN_LEFT,
            color = {1, 0.5, 0, 1}
        })
        return
    end

    -- System status
    table.insert(self.ui.appTexts, {
        text = "System: " .. diggingInfo.terrainSystem,
        x = content.x + padX,
        y = yPos,
        size = 0.016,
        align = RenderText.ALIGN_LEFT,
        color = {0.4, 0.8, 0.4, 1}
    })

    -- TerraFarm detection
    if diggingInfo.supportsTerraFarm then
        yPos = yPos - 0.024
        table.insert(self.ui.appTexts, {
            text = "✓ TerraFarm Mod Detected",
            x = content.x + padX,
            y = yPos,
            size = 0.016,
            align = RenderText.ALIGN_LEFT,
            color = {0.4, 0.9, 0.4, 1}
        })
        
        if diggingInfo.terraFarmVersion then
            table.insert(self.ui.appTexts, {
                text = "Version: " .. diggingInfo.terraFarmVersion,
                x = content.x + content.width - padX,
                y = yPos,
                size = 0.014,
                align = RenderText.ALIGN_RIGHT,
                color = {0.7, 0.7, 0.7, 1}
            })
        end
    end

    -- Digging statistics
    yPos = yPos - 0.030
    table.insert(self.ui.appTexts, {
        text = "Digging Tools: " .. diggingInfo.diggingTools,
        x = content.x + padX,
        y = yPos,
        size = 0.016,
        align = RenderText.ALIGN_LEFT,
        color = self.UI_CONSTANTS.TEXT_COLOR
    })

    table.insert(self.ui.appTexts, {
        text = "Active: " .. diggingInfo.activeDiggers,
        x = content.x + content.width - padX,
        y = yPos,
        size = 0.016,
        align = RenderText.ALIGN_RIGHT,
        color = {0.4, 0.8, 0.4, 1}
    })

    -- Player position
    if diggingInfo.currentPosition then
        yPos = yPos - 0.024
        table.insert(self.ui.appTexts, {
            text = string.format("Position: X %.1f  Z %.1f",
                diggingInfo.currentPosition.x,
                diggingInfo.currentPosition.z),
            x = content.x + padX,
            y = yPos,
            size = 0.015,
            align = RenderText.ALIGN_LEFT,
            color = {0.8, 0.8, 0.8, 1}
        })

        if diggingInfo.currentTerrainHeight then
            yPos = yPos - 0.020
            table.insert(self.ui.appTexts, {
                text = string.format("Height: %.2f m", diggingInfo.currentTerrainHeight),
                x = content.x + padX,
                y = yPos,
                size = 0.015,
                align = RenderText.ALIGN_LEFT,
                color = {0.8, 0.8, 0.8, 1}
            })
        end

        -- Terrain delta (cut/fill)
        if diggingInfo.terrainDelta then
            yPos = yPos - 0.020
            local delta = diggingInfo.terrainDelta
            local deltaText, deltaColor
            
            if delta < 0 then
                deltaText = string.format("Cut: %.2f m", math.abs(delta))
                deltaColor = {0.9, 0.7, 0.4, 1}  -- Orange for cut
            elseif delta > 0 then
                deltaText = string.format("Fill: %.2f m", delta)
                deltaColor = {0.4, 0.7, 0.9, 1}  -- Blue for fill
            else
                deltaText = "No change"
                deltaColor = {0.7, 0.7, 0.7, 1}
            end
            
            table.insert(self.ui.appTexts, {
                text = deltaText,
                x = content.x + padX,
                y = yPos,
                size = 0.015,
                align = RenderText.ALIGN_LEFT,
                color = deltaColor
            })
        end
    end

    -- TerraFarm vehicles
    if diggingInfo.supportsTerraFarm and #diggingInfo.terraFarmVehicles > 0 then
        yPos = yPos - 0.030
        table.insert(self.ui.appTexts, {
            text = "TerraFarm Vehicles:",
            x = content.x + padX,
            y = yPos,
            size = 0.016,
            align = RenderText.ALIGN_LEFT,
            color = {0.3, 0.6, 0.8, 1}
        })

        yPos = yPos - 0.022
        for i = 1, math.min(3, #diggingInfo.terraFarmVehicles) do
            local vehicle = diggingInfo.terraFarmVehicles[i]
            table.insert(self.ui.appTexts, {
                text = "• " .. (vehicle.name or "Unknown"),
                x = content.x + padX + 0.01,
                y = yPos,
                size = 0.013,
                align = RenderText.ALIGN_LEFT,
                color = {0.8, 0.8, 0.8, 1}
            })

            table.insert(self.ui.appTexts, {
                text = vehicle.isActive and "ACTIVE" or "Idle",
                x = content.x + content.width - padX,
                y = yPos,
                size = 0.013,
                align = RenderText.ALIGN_RIGHT,
                color = vehicle.isActive and {0.4, 0.9, 0.4, 1} or {0.7, 0.7, 0.7, 1}
            })

            yPos = yPos - 0.018
        end
    end

    -- All digging vehicles
    if diggingInfo.availableTools and #diggingInfo.availableTools > 0 then
        yPos = yPos - 0.030
        table.insert(self.ui.appTexts, {
            text = "All Digging Vehicles:",
            x = content.x + padX,
            y = yPos,
            size = 0.016,
            align = RenderText.ALIGN_LEFT,
            color = {0.3, 0.6, 0.8, 1}
        })

        yPos = yPos - 0.022
        for i = 1, math.min(5, #diggingInfo.availableTools) do
            local tool = diggingInfo.availableTools[i]
            table.insert(self.ui.appTexts, {
                text = "• " .. (tool.name or "Unknown"),
                x = content.x + padX + 0.01,
                y = yPos,
                size = 0.013,
                align = RenderText.ALIGN_LEFT,
                color = {0.8, 0.8, 0.8, 1}
            })

            local statusText = tool.status or "Idle"
            if tool.type ~= "Unknown" then
                statusText = tool.type .. " - " .. statusText
            end
            
            table.insert(self.ui.appTexts, {
                text = statusText,
                x = content.x + content.width - padX,
                y = yPos,
                size = 0.013,
                align = RenderText.ALIGN_RIGHT,
                color = (tool.status == "ACTIVE") and {0.4, 0.9, 0.4, 1} or {0.7, 0.7, 0.7, 1}
            })

            yPos = yPos - 0.018
        end
    end

    self:log("Digging app loaded successfully")
end

function FarmTabletUI:getDiggingInfo()
    local info = {
        hasTerrainSystem = false,
        terrainSystem = "Unknown",
        supportsTerraFarm = false,
        terraFarmActive = false,
        
        diggingTools = 0,
        activeDiggers = 0,
        
        supportsTerrainDeformation = false,
        terrainDeformationActive = false,
        
        currentPosition = nil,
        currentTerrainHeight = nil,
        terrainDelta = nil,
        
        availableTools = {},
        terraFarmVehicles = {}
    }
    
    -- Check for TerraFarm mod (FS25 might have it integrated)
    if g_terraFarm ~= nil then
        info.supportsTerraFarm = true
        info.terraFarmActive = true
        info.terrainSystem = "TerraFarm"
        
        -- Get TerraFarm specific info
        if g_terraFarm.getVersion then
            info.terraFarmVersion = g_terraFarm:getVersion()
        end
        
        -- Check for active TerraFarm vehicles
        if g_currentMission and g_currentMission.vehicles then
            for _, vehicle in pairs(g_currentMission.vehicles) do
                if vehicle.spec_terraFarm ~= nil then
                    table.insert(info.terraFarmVehicles, {
                        name = vehicle.getName and vehicle:getName() or "Unknown",
                        isActive = self:isVehicleDigging(vehicle)
                    })
                end
            end
        end
    end
    
    -- Check for standard FS25 terrain system
    if g_currentMission and g_currentMission.terrainRootNode then
        info.hasTerrainSystem = true
        info.terrainSystem = "FS25 Terrain"
        
        if g_currentMission.terrainDeformationSystem then
            info.supportsTerrainDeformation = true
            info.terrainDeformationActive = true
        end
    end
    
    -- Detect digging vehicles
    if g_currentMission and g_currentMission.vehicles then
        for _, vehicle in pairs(g_currentMission.vehicles) do
            if self:isDiggingCapable(vehicle) then
                info.diggingTools = info.diggingTools + 1
                
                local active = self:isVehicleDigging(vehicle)
                if active then
                    info.activeDiggers = info.activeDiggers + 1
                end
                
                local vehicleType = "Unknown"
                if vehicle.spec_terraFarm then
                    vehicleType = "TerraFarm"
                elseif vehicle.spec_digging then
                    vehicleType = "Digging"
                elseif vehicle.typeName then
                    vehicleType = vehicle.typeName
                end
                
                table.insert(info.availableTools, {
                    name = vehicle.getName and vehicle:getName() or "Unknown",
                    type = vehicleType,
                    status = active and "ACTIVE" or "Idle"
                })
            end
        end
    end
    
    -- Get player position and terrain data
    if g_currentMission and g_currentMission.player then
        local player = g_currentMission.player
        if player.rootNode and g_currentMission.terrainRootNode then
            local x, _, z = getWorldTranslation(player.rootNode)
            info.currentPosition = { x = x, z = z }
            
            -- Get terrain height
            local currentH = getTerrainHeightAtWorldPos(
                g_currentMission.terrainRootNode,
                x, 0, z
            )
            info.currentTerrainHeight = currentH
            
            -- Cache original terrain height for delta calculation
            self._terrainCache = self._terrainCache or {}
            local key = string.format("%.1f_%.1f", x, z)
            
            if self._terrainCache[key] == nil then
                self._terrainCache[key] = currentH
            end
            
            info.terrainDelta = currentH - self._terrainCache[key]
        end
    end
    
    return info
end

function FarmTabletUI:isDiggingCapable(vehicle)
    -- Check for standard FS25 terrain deformation
    if vehicle ~= nil and vehicle.getIsTerrainDeformationActive ~= nil then
        return true
    end
    
    -- Check for digging attachments
    if vehicle.getAttachedImplements then
        local attached = vehicle:getAttachedImplements()
        for _, impl in ipairs(attached) do
            if impl.object.spec_digging or impl.object.spec_terraFarm then
                return true
            end
        end
    end
    
    -- Check vehicle type name for digging indicators
    if vehicle ~= nil then
        local typeName = vehicle.typeName or ""
        typeName = typeName:lower()
        
        return typeName:find("digger") or 
               typeName:find("excavator") or
               typeName:find("backhoe") or
               typeName:find("loader") or
               typeName:find("terra")
    end
    
    return false
end

function FarmTabletUI:isVehicleDigging(vehicle)
    -- Standard FS25 terrain deformation
    if vehicle.getIsTerrainDeformationActive ~= nil then
        return vehicle:getIsTerrainDeformationActive()
    end
    
    -- TerraFarm specific detection
    if vehicle.spec_terraFarm ~= nil then
        local terraSpec = vehicle.spec_terraFarm
        if terraSpec.isActive ~= nil then
            return terraSpec.isActive
        end
        
        -- Check fill type changes
        if terraSpec.fillLevel ~= nil and terraSpec.fillLevel > 0 then
            return true
        end
    end
    
    -- Check digging attachments
    if vehicle.getAttachedImplements then
        local attached = vehicle:getAttachedImplements()
        for _, impl in ipairs(attached) do
            local implObject = impl.object
            
            -- Check if implement has digging functionality
            if implObject.spec_digging then
                local diggingSpec = implObject.spec_digging
                if diggingSpec.isActive ~= nil then
                    return diggingSpec.isActive
                end
            end
            
            -- Check for TerraFarm attachment
            if implObject.spec_terraFarm then
                local terraSpec = implObject.spec_terraFarm
                if terraSpec.isActive ~= nil then
                    return terraSpec.isActive
                end
            end
        end
    end
    
    -- Check vehicle animation/working state
    if vehicle.getIsWorkAreaActive ~= nil then
        return vehicle:getIsWorkAreaActive()
    end
    
    return false
end