-- =========================================================
-- Workshop App - Quick access to vehicle workshop (FS25)
-- =========================================================

function FarmTabletUI:loadWorkshopApp()
    local content = self.ui.appContentArea
    if not content then return end

    local padX = self:px(15)
    local padY = self:py(15)
    local centerX = content.x + content.width * 0.5
    local centerY = content.y + content.height * 0.5

    -- Title
    table.insert(self.ui.appTexts, {
        text = "Vehicle Workshop",
        x = content.x + padX,
        y = content.y + content.height - padY - 0.035,
        size = 0.022,
        align = RenderText.ALIGN_LEFT,
        color = self.UI_CONSTANTS.TEXT_COLOR
    })

    -- Subtitle
    table.insert(self.ui.appTexts, {
        text = "Open workshop for the targeted vehicle",
        x = content.x + padX,
        y = content.y + content.height - padY - 0.080,
        size = 0.016,
        align = RenderText.ALIGN_LEFT,
        color = {0.8, 0.8, 0.8, 1}
    })

    -- Button
    local btnW = self:px(200)
    local btnH = self:py(45)
    local btnX = centerX - btnW * 0.5
    local btnY = centerY - btnH * 0.5

    local btn = self:createBlankOverlay(
        btnX,
        btnY,
        btnW,
        btnH,
        {0.3, 0.6, 0.3, 0.9}
    )
    btn:setVisible(true)

    table.insert(self.ui.overlays, btn)
    table.insert(self.ui.contentOverlays, btn)

    self.ui.workshopButton = {
        overlay = btn,
        x = btnX,
        y = btnY,
        width = btnW,
        height = btnH
    }

    table.insert(self.ui.appTexts, {
        text = "Open Workshop",
        x = centerX,
        y = btnY + btnH * 0.5 - 0.006,
        size = 0.018,
        align = RenderText.ALIGN_CENTER,
        color = {1, 1, 1, 1}
    })

    -- Vehicle info area - this will be updated dynamically
    self.ui.workshopVehicleText = {
        text = "",
        x = centerX,
        y = btnY - 0.040,
        size = 0.014,
        align = RenderText.ALIGN_CENTER,
        color = {1, 0.5, 0, 1} -- Orange for no vehicle
    }
    table.insert(self.ui.appTexts, self.ui.workshopVehicleText)

    -- Initialize update tracking
    self.workshopLastUpdate = 0
    self.workshopUpdateInterval = 500 -- Update every 500ms
    
    self:log("Workshop app loaded")
end

-- =========================================================
-- Vehicle detection (FS25-native, updated version)
-- =========================================================

function FarmTabletUI:getTargetedWorkshopVehicle()
    if not g_currentMission or not g_currentMission.player then
        return nil
    end

    local player = g_currentMission.player
    local farmId = g_currentMission:getFarmId()

    if farmId == FarmManager.SPECTATOR_FARM_ID then
        return nil
    end

    -- Try different targeting methods
    local targetNode = nil
    
    -- Method 1: Check if player has a targeter directly
    if player.targeter ~= nil then
        -- Try to get any targeted object
        targetNode = player.targeter:getClosestTargetedNode()
        
        -- If that doesn't work, check raycast results
        if targetNode == nil and player.targeter.raycastNode ~= nil then
            targetNode = player.targeter.raycastNode
        end
    end
    
    -- Method 2: Check for raycast collision
    if targetNode == nil and player.raycastCollisionNode ~= nil then
        targetNode = player.raycastCollisionNode
    end

    -- Method 3: Try the original MobileWorkshop approach
    if targetNode == nil and player.targeter ~= nil then
        targetNode = player.targeter:getClosestTargetedNodeFromType(PlayerInputComponent)
    end

    if targetNode ~= nil and entityExists(targetNode) then
        local object = g_currentMission:getNodeObject(targetNode)

        if object ~= nil
        and object:isa(Vehicle)
        and object:getShowInVehiclesOverview()
        and object:getOwnerFarmId() == farmId then
            return object.rootVehicle or object
        end
    end

    return nil
end

-- =========================================================
-- Update workshop app (called from update function)
-- =========================================================

function FarmTabletUI:updateWorkshopApp(dt)
    if not self.isTabletOpen or self.tabletSystem.currentApp ~= "workshop" then
        return
    end
    
    -- Update vehicle info at intervals
    self.workshopLastUpdate = (self.workshopLastUpdate or 0) + dt
    
    if self.workshopLastUpdate * 1000 >= self.workshopUpdateInterval then
        self.workshopLastUpdate = 0
        
        local vehicle = self:getTargetedWorkshopVehicle()
        
        if vehicle and self.ui.workshopVehicleText then
            self.ui.workshopVehicleText.text = "Targeted: " .. vehicle:getFullName()
            self.ui.workshopVehicleText.color = {0.4, 0.9, 0.4, 1} -- Green for vehicle found
        elseif self.ui.workshopVehicleText then
            self.ui.workshopVehicleText.text = "No valid vehicle targeted"
            self.ui.workshopVehicleText.color = {1, 0.5, 0, 1} -- Orange for no vehicle
        end
        
        if self.settings.debugMode then
            print("[Farm Tablet Workshop] Updated vehicle info: " .. (vehicle and vehicle:getFullName() or "None"))
        end
    end
end

-- =========================================================
-- Open workshop (FS25-correct flow)
-- =========================================================

function FarmTabletUI:openWorkshopForNearestVehicle(distance)
    local vehicle = self:getTargetedWorkshopVehicle()

    if vehicle == nil then
        -- Try to find any nearby vehicle within distance
        if distance and g_currentMission and g_currentMission.player then
            local player = g_currentMission.player
            local playerX, playerY, playerZ = getWorldTranslation(player.rootNode)
            local farmId = g_currentMission:getFarmId()
            
            local minDistance = math.huge
            local closestVehicle = nil
            
            for _, object in pairs(g_currentMission.vehicles or {}) do
                if object:isa(Vehicle) 
                and object:getShowInVehiclesOverview()
                and object:getOwnerFarmId() == farmId then
                    local vX, vY, vZ = getWorldTranslation(object.rootNode)
                    local dist = MathUtil.vector3Length(vX - playerX, vY - playerY, vZ - playerZ)
                    
                    if dist < distance and dist < minDistance then
                        minDistance = dist
                        closestVehicle = object
                    end
                end
            end
            
            vehicle = closestVehicle
        end
        
        if vehicle == nil then
            if g_currentMission and g_currentMission.hud then
                g_currentMission.hud:showBlinkingWarning(
                    "No valid vehicle targeted",
                    4000
                )
            end
            return false
        end
    end

    if not g_workshopScreen then
        self:log("Workshop screen not available")
        return false
    end

    -- Collect vehicles exactly how FS25 / MobileWorkshop does
    local vehicles = {}
    local farmId = g_currentMission:getFarmId()

    for _, subVehicle in ipairs(vehicle.rootVehicle:getChildVehicles()) do
        if subVehicle:getShowInVehiclesOverview()
        and subVehicle:getOwnerFarmId() == farmId then
            table.insert(vehicles, subVehicle)
        end
    end

    if #vehicles == 0 then
        self:log("No vehicles found in the targeted vehicle group")
        return false
    end

    table.sort(vehicles, function(a, b)
        return a.rootNode < b.rootNode
    end)

    -- Close tablet BEFORE opening workshop
    self:closeTablet()

    -- Open workshop
    g_workshopScreen:setSellingPoint(nil, false, false, true)
    g_workshopScreen:setVehicles(vehicles)

    -- SAFETY: list might not exist yet
    if g_workshopScreen.list ~= nil then
        local selectedIndex = table.find(vehicles, vehicle)
        if selectedIndex then
            g_workshopScreen.list:setSelectedIndex(selectedIndex)
        end
    end

    g_gui:showGui("WorkshopScreen")
    
    self:log("Opened workshop for vehicle: " .. vehicle:getFullName())
    return true
end