--[[
Copyright (C) GtX (Andy), 2022

Author: GtX | Andy
Date: 09.10.2022
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

MoveHusbandryAnimals = {}
MoveHusbandryAnimals.TRANSPORT_DISTANCE_FACTOR = 50

local MoveHusbandryAnimals_mt = Class(MoveHusbandryAnimals)

local modName = g_currentModName
local modDirectory = g_currentModDirectory

local versionString = "0.0.0.0"
local buildId = 1

local validationFail

function MoveHusbandryAnimals.new(numInserted)
    local self = setmetatable({}, MoveHusbandryAnimals_mt)

    self.modName = modName

    self.versionString = versionString
    self.buildId = buildId

    self.updateTime = 1000
    self.gameStarted = false

    self.transportFee = 12
    self.transportDistance = 300

    self.settings = {}
    self:createSettings()

    self.numInserted = numInserted or 0

    g_messageCenter:subscribe(MessageType.MOVE_HUSBANDRY_ANIMALS_SETTINGS_CHANGED, self.onSettingsChanged, self)
    g_messageCenter:subscribe(MessageType.CURRENT_MISSION_START, self.onMissionStarted, self)

    return self
end

function MoveHusbandryAnimals:createSettings()
    for i = #self.settings, 1, -1 do
        g_messageCenter:unsubscribeAll(self.settings[i])
        table.remove(self.settings, i)
    end

    local transportFeeSetting = {
        name = "transportFee",
        id = "moveHusbandryAnimals_multiTransportFee",
        title = g_i18n:getText("setting_transportFee", modName),
        toolTip = g_i18n:getText("toolTip_transportFee", modName),
        maxState = 201,
        state = 13,
        numBits = 8
    }

    function transportFeeSetting.setState(setting, state, suppressInfo)
        setting.state = math.clamp(state, 1, setting.maxState)

        local transportFee = setting.state - 1

        if transportFee ~= self.transportFee then
            self.transportFee = transportFee

            if not suppressInfo and self.gameStarted then
                Logging.info("MoveHusbandryAnimals Setting '%s': %s", setting.name, transportFee)
            end
        end
    end

    function transportFeeSetting.setTexts(setting)
        setting.maxState = 201
        setting.numBits = MathUtil.getNumRequiredBits(setting.maxState)

        if setting.element ~= nil then
            local texts = table.create(setting.maxState)

            for i = 0, setting.maxState - 1 do
                table.insert(texts, g_i18n:formatMoney(i, 0, true, false))
            end

            setting.element:setTexts(texts)
        end
    end

    g_messageCenter:subscribe(MessageType.SETTING_CHANGED[GameSettings.SETTING.MONEY_UNIT], transportFeeSetting.setTexts, transportFeeSetting)

    self.transportFeeSetting = transportFeeSetting
    table.insert(self.settings, transportFeeSetting)

    local transportDistanceSetting = {
        name = "transportDistance",
        id = "moveHusbandryAnimals_multiTransportDistance",
        title = g_i18n:getText("setting_transportDistance", modName),
        toolTip = g_i18n:getText("toolTip_transportDistance", modName),
        maxState = 41,
        state = 7,
        numBits = 6
    }

    function transportDistanceSetting.setState(setting, state, suppressInfo)
        setting.state = math.clamp(state, 1, setting.maxState)

        local transportDistance = (setting.state - 1) * MoveHusbandryAnimals.TRANSPORT_DISTANCE_FACTOR

        if transportDistance ~= self.transportDistance then
            self.transportDistance = transportDistance

            if not suppressInfo and self.gameStarted then
                Logging.info("MoveHusbandryAnimals Setting '%s': %s", setting.name, transportDistance)
            end
        end
    end

    function transportDistanceSetting.setTexts(setting)
        local terrainSize = 2048

        if g_currentMission and g_currentMission.terrainSize then
            terrainSize = g_currentMission.terrainSize
        end

        setting.maxState = terrainSize > 2048 and 81 or 41
        setting.numBits = MathUtil.getNumRequiredBits(setting.maxState)

        if setting.element ~= nil then
            local useFahrenheit = g_i18n.useFahrenheit
            local distanceText = " " .. g_i18n:getText(useFahrenheit and "unit_ftShort" or "unit_mShort")

            local function getDistanceText(distance)
                if useFahrenheit then
                    return MathUtil.round(distance * 3.28084, 0) .. distanceText
                end

                return MathUtil.round(distance, 0) .. distanceText
            end

            local texts = table.create(setting.maxState)

            for i = 0, setting.maxState - 1 do
                table.insert(texts, getDistanceText(i * MoveHusbandryAnimals.TRANSPORT_DISTANCE_FACTOR))
            end

            setting.element:setTexts(texts)
        end
    end

    g_messageCenter:subscribe(MessageType.SETTING_CHANGED[GameSettings.SETTING.USE_FAHRENHEIT], transportDistanceSetting.setTexts, transportDistanceSetting)

    self.transportDistanceSetting = transportDistanceSetting
    table.insert(self.settings, transportDistanceSetting)
end

function MoveHusbandryAnimals:delete()
    self.gameStarted = false

    g_messageCenter:unsubscribeAll(self)

    for _, setting in ipairs (self.settings) do
        g_messageCenter:unsubscribeAll(setting)
    end

    if g_currentMission ~= nil then
        g_currentMission:removeUpdateable(self)
    end
end

function MoveHusbandryAnimals:update(dt)
    if self.settingsQueued and self.gameStarted then
        self.updateTime -= dt

        if self.updateTime <= 0 then
            for _, setting in ipairs (self.settings) do
                if setting.queuedState ~= nil then
                    setting:setState(setting.queuedState)
                    setting.queuedState = nil
                end
            end

            MoveAnimalsSettingsEvent.sendEvent()
            self.settingsQueued = false
        end
    else
        g_currentMission:removeUpdateable(self)
    end
end

function MoveHusbandryAnimals:saveToXMLFile(xmlFile, key)
    xmlFile:setString(string.format("%s#version", key), self.versionString)
    xmlFile:setFloat(string.format("%s#buildId", key), self.buildId)

    for _, setting in ipairs (self.settings) do
        xmlFile:setInt(string.format("%s.%s#state", key, setting.name), setting.state)
    end
end

function MoveHusbandryAnimals:loadFromXMLFile(xmlFile, key)
    for _, setting in ipairs (self.settings) do
        setting:setState(xmlFile:getInt(string.format("%s.%s#state", key, setting.name), setting.state), true)
    end
end

function MoveHusbandryAnimals:queueSettingStateChange(setting, state)
    if setting ~= nil and self.gameStarted then
        setting.queuedState = state or setting.state

        self.updateTime = 1000
        self.settingsQueued = true

        g_currentMission:addUpdateable(self)
    end
end

function MoveHusbandryAnimals:onMissionStarted(isNewSavegame)
    for _, setting in ipairs (self.settings) do
        local settingValue = self[setting.name]

        if settingValue ~= nil then
            Logging.info("MoveHusbandryAnimals Setting '%s': %s", setting.name, settingValue)
        end
    end

    self.gameStarted = true
end

function MoveHusbandryAnimals:onSettingsChanged()
    for _, setting in ipairs (self.settings) do
        if setting.element ~= nil then
            setting.element:setState(setting.state)
        end
    end
end

function MoveHusbandryAnimals:getTransportCosts(numAnimals, distance)
    if distance ~= nil and distance >= self.transportDistance then
        return self.transportFee * (numAnimals or 0), self.transportFee, true
    end

    return 0, self.transportFee, false
end

local function isActive()
    return g_modIsLoaded[modName] and g_moveHusbandryAnimals ~= nil
end

local function validateMod()
    local mod = g_modManager:getModByName(modName)

    if mod == nil or g_iconGenerator ~= nil or g_isEditor then
        return false
    end

    versionString = mod.version or versionString

    if mod.modName == "FS25_MoveHusbandryAnimals" or mod.modName == "FS25_MoveHusbandryAnimals_update" then
        if mod.author ~= nil and #mod.author == 3 then
            return true
        end
    end

    validationFail = {
        startUpdateTime = 2000,

        update = function(self, dt)
            self.startUpdateTime = self.startUpdateTime - dt

            if self.startUpdateTime < 0 then
                local text = string.namedFormat(g_i18n:getText("ui_loadError", mod.modName), "modName", mod.modName, "author", mod.author or "Unknown")

                if g_dedicatedServer == nil then
                    if not g_gui:getIsGuiVisible() then
                        local title = string.format("%s - Version %s", mod.title, versionString)
                        local yesText = g_i18n:getText("button_modHubDownload")
                        local noText = g_i18n:getText("button_ok")

                        YesNoDialog.show(self.openModHubLink, nil, text, title, yesText, noText, DialogElement.TYPE_LOADING)
                    end
                else
                    print("\n" .. text .. "\n    - https://farming-simulator.com/mods.php?&title=fs2025&filter=org&org_id=129652&page=0" .. "\n")
                    self.openModHubLink(false)
                end
            end
        end,

        openModHubLink = function(yes)
            if yes then
                openWebFile("mods.php?title=fs2025&filter=org&org_id=129652&page=0", "")
            end

            removeModEventListener(validationFail)
            validationFail = nil
        end
    }

    addModEventListener(validationFail)

    return false
end

local function validateTypes(typeManager)
    if typeManager.typeName == "placeable" and g_moveHusbandryAnimals == nil then
        local specializationName = PlaceableHusbandryMoveAnimals.SPEC_NAME
        local specializationObject = g_placeableSpecializationManager:getSpecializationObjectByName(specializationName)

        if specializationObject ~= nil then
            local numInserted = 0

            for typeName, typeEntry in pairs (typeManager:getTypes()) do
                if specializationObject.prerequisitesPresent(typeEntry.specializations) then
                    typeManager:addSpecialization(typeName, specializationName)

                    numInserted += 1
                end
            end

            if numInserted > 0 then
                g_moveHusbandryAnimals = MoveHusbandryAnimals.new(numInserted)

                MoveAnimalsScreen.register()
            end
        end
    end
end

local function unloadMapData()
    if g_moveHusbandryAnimals ~= nil then
        g_moveHusbandryAnimals:delete()
    end

    g_moveHusbandryAnimals = nil
end

local function updateGameSettings(frame)
    if not isActive() then
        return
    end

    if frame.moveHusbandryAnimals_sectionHeader == nil then
        local gameSettingsLayout = frame.gameSettingsLayout

        for i, element in ipairs (gameSettingsLayout.elements) do
            if element:isa(TextElement) then
                frame.moveHusbandryAnimals_sectionHeader = element:clone(gameSettingsLayout)
                frame.moveHusbandryAnimals_sectionHeader:setText(g_i18n:getText("ui_moveHusbandryAnimals", modName))

                break
            end
        end

        local multiTextOptionParent = nil

        for i, element in ipairs (gameSettingsLayout.elements) do
            if #element.elements > 0 and element:isa(BitmapElement) then
                if element.elements[1]:isa(MultiTextOptionElement) then
                    multiTextOptionParent = element

                    break
                end
            end
        end

        if multiTextOptionParent ~= nil then
            for _, setting in ipairs (g_moveHusbandryAnimals.settings) do
                local parent = multiTextOptionParent:clone(gameSettingsLayout, false)
                local multiTextOption = parent.elements[1]
                local multiTextOptionDelete = multiTextOption.delete

                function multiTextOption.delete(element)
                    setting.element = nil
                    multiTextOptionDelete(element)
                end

                function multiTextOption.onClickCallback(_, state)
                    if g_moveHusbandryAnimals ~= nil then
                        g_moveHusbandryAnimals:queueSettingStateChange(setting, state)
                    end
                end

                parent.elements[2]:setText(setting.title)
                multiTextOption.elements[1]:setText(setting.toolTip)

                multiTextOption.id = setting.id

                multiTextOption:setVisible(true)
                multiTextOption:setDisabled(false)

                frame[setting.id] = multiTextOption
                setting.element = multiTextOption

                setting:setTexts()

                parent:setVisible(true)
                parent:setDisabled(false)
            end
        end

        gameSettingsLayout:invalidateLayout()
    end

    g_moveHusbandryAnimals:onSettingsChanged()
end

local function sendInitialClientState(_, connection, user, farm)
    if isActive() then
        connection:sendEvent(MoveAnimalsSettingsEvent.new())
    end
end

local function saveItems()
    if isActive() then
        local xmlFilename = g_currentMission.missionInfo.savegameDirectory .. "/moveHusbandryAnimals.xml"
        local xmlFile = XMLFile.create("moveHusbandryAnimalsXML", xmlFilename, "moveHusbandryAnimals")

        if xmlFile ~= nil then
            g_moveHusbandryAnimals:saveToXMLFile(xmlFile, "moveHusbandryAnimals")

            xmlFile:save()
            xmlFile:delete()
        end
    end
end

local function loadItems()
    if isActive() and g_currentMission.missionInfo.savegameDirectory ~= nil then
        g_asyncTaskManager:addTask(function()
            local xmlFilename = g_currentMission.missionInfo.savegameDirectory .. "/moveHusbandryAnimals.xml"
            local xmlFile = XMLFile.loadIfExists("moveHusbandryAnimalsXML", xmlFilename)

            if xmlFile ~= nil then
                g_moveHusbandryAnimals:loadFromXMLFile(xmlFile, "moveHusbandryAnimals")

                xmlFile:delete()
            end
        end)
    end
end

local function init()
    if validateMod() then
        MessageType.MOVE_HUSBANDRY_ANIMALS_SETTINGS_CHANGED = nextMessageTypeId()

        source(modDirectory .. "scripts/events/MoveAnimalsSettingsEvent.lua")
        source(modDirectory .. "scripts/events/MoveAnimalsEvent.lua")
        source(modDirectory .. "scripts/gui/MoveAnimalsScreen.lua")

        g_placeableSpecializationManager:addSpecialization("husbandryMoveAnimals", "PlaceableHusbandryMoveAnimals", modDirectory .. "scripts/placeables/PlaceableHusbandryMoveAnimals.lua", nil)

        TypeManager.validateTypes = Utils.prependedFunction(TypeManager.validateTypes, validateTypes)
        TypeManager.unloadMapData = Utils.appendedFunction(TypeManager.unloadMapData, unloadMapData)

        InGameMenuSettingsFrame.updateGameSettings = Utils.appendedFunction(InGameMenuSettingsFrame.updateGameSettings, updateGameSettings)
        FSBaseMission.sendInitialClientState = Utils.appendedFunction(FSBaseMission.sendInitialClientState, sendInitialClientState)

        ItemSystem.save = Utils.prependedFunction(ItemSystem.save, saveItems)
        ItemSystem.loadItems = Utils.prependedFunction(ItemSystem.loadItems, loadItems)
    else
        Logging.error("[%s] Failed to initialise / validate mod!", modName)
    end
end

init()
