-- =========================================================
-- Author: U_BMP
-- Group: https://vk.com/biomodprod_utilit_fs
-- Date: 19.11.2025
--
-- Console:
--   gsWheelDbg  [0|1] [segments] [range]
--
-- Notes:
--  - Dedicated server: nothing will draw (no draw pass).
--  - If HUD shows "inVeh=YES" but wheelsDrawn=0 => wheel nodes not detected -> we’ll extend selector.
-- =========================================================

WheelInspectorDebug = WheelInspectorDebug or {}
WheelInspectorDebug.name = (g_currentModName or "WheelInspectorDebug")
WheelInspectorDebug.path = (g_currentModDirectory or "")

addModEventListener(WheelInspectorDebug)

WheelInspectorDebug.enabled  = false
WheelInspectorDebug.segments = 28
WheelInspectorDebug.maxRange = 5000

WheelInspectorDebug._registered = false
WheelInspectorDebug._drawHooked = false

WheelInspectorDebug.terrainSampleIntervalMs = 200
WheelInspectorDebug._terrainCache = {}

WheelInspectorDebug.textSmoothTauSec = 0.15
WheelInspectorDebug.textDeadzone    = 0.015
WheelInspectorDebug._textPosCache   = {}

-- ---------------------------------------------------------
-- utils
-- ---------------------------------------------------------
local function clamp(v, a, b)
	v = tonumber(v)
	if v == nil then return a end
	if v < a then return a end
	if v > b then return b end
	return v
end

local function toBool01(v)
	if v == nil then return nil end
	if v == true or v == "1" or v == 1 or v == "true" or v == "on" then return true end
	if v == false or v == "0" or v == 0 or v == "false" or v == "off" then return false end
	return nil
end

local function fmtNum(v, prec)
	if v == nil then return "n/a" end
	if type(v) ~= "number" then return tostring(v) end
	prec = prec or 3
	return string.format("%." .. tostring(prec) .. "f", v)
end

local function distSq(x1,y1,z1, x2,y2,z2)
	local dx = x1-x2
	local dy = y1-y2
	local dz = z1-z2
	return dx*dx + dy*dy + dz*dz
end

local function norm(x,y,z)
	local l = math.sqrt(x*x + y*y + z*z)
	if l < 1e-6 then
		return 0,0,0
	end
	return x/l, y/l, z/l
end

local function isValidNode(n)
	if n == nil or n == 0 then return false end
	if type(entityExists) == "function" then
		return entityExists(n)
	end
	return true
end

local function getCamNode()
	return (g_camera ~= nil) and g_camera.node or nil
end

local function getCamWorldPos()
	local cam = getCamNode()
	if cam ~= nil then
		return getWorldTranslation(cam)
	end
	return 0,0,0
end

local function drawLine(x1,y1,z1, x2,y2,z2, r,g,b, solid)
	drawDebugLine(x1,y1,z1, r,g,b, x2,y2,z2, r,g,b, solid == true)
end

local function drawWorldText(x,y,z, text)
	if text == nil or text == "" then return end
	if Utils == nil or Utils.renderTextAtWorldPosition == nil then return end
	Utils.renderTextAtWorldPosition(
		x, y, z,
		text,
		getCorrectTextSize(0.012),
		0,
		1,1,1,1
	)
end

local function drawHudLine(line, y)
	setTextAlignment(RenderText.ALIGN_LEFT)
	setTextColor(1,1,1,1)
	renderText(0.012, y, getCorrectTextSize(0.017), line)
end

-- ---------------------------------------------------------
-- ground / terrain helpers
-- ---------------------------------------------------------
local function getFieldGroundTypeNameByValue(v)
	if v == nil then return "n/a" end
	if FieldGroundType ~= nil then
		for k, vv in pairs(FieldGroundType) do
			if type(k) == "string" and type(vv) == "number" and vv == v then
				return k
			end
		end
	end
	return tostring(v)
end

local function getTerrainLayerNameSafe(layerId)
	if layerId == nil then return nil end
	if g_terrainNode == nil then return nil end
	if type(getTerrainNumOfLayers) ~= "function" or type(getTerrainLayerName) ~= "function" then
		return nil
	end

	local num = getTerrainNumOfLayers(g_terrainNode) or 0
	if type(layerId) == "number" and layerId >= 0 and layerId < num then
		return getTerrainLayerName(g_terrainNode, layerId)
	end
	return nil
end

local function getWheelContactWorldPos(wheelNode, p)
	if p ~= nil and p.lastContactX ~= nil and p.lastContactY ~= nil and p.lastContactZ ~= nil then
		return p.lastContactX, p.lastContactY, p.lastContactZ
	end
	if wheelNode ~= nil and isValidNode(wheelNode) then
		return getWorldTranslation(wheelNode)
	end
	return 0,0,0
end

local function getTimeMs()
	if g_time ~= nil then
		return g_time
	end
	return math.floor(os.clock() * 1000)
end

local function smoothWorldPosForWheel(wheel, x, y, z, dtMs)
	if wheel == nil then return x, y, z end

	local cache = WheelInspectorDebug._textPosCache[wheel]
	if cache == nil then
		cache = { x = x, y = y, z = z }
		WheelInspectorDebug._textPosCache[wheel] = cache
		return x, y, z
	end

	local dx, dy, dz = x - cache.x, y - cache.y, z - cache.z
	local d2 = dx*dx + dy*dy + dz*dz
	local dz0 = (WheelInspectorDebug.textDeadzone or 0)
	if dz0 > 0 and d2 < dz0*dz0 then
		return cache.x, cache.y, cache.z
	end

	local tau = WheelInspectorDebug.textSmoothTauSec or 0.12
	local dtS = (dtMs or 0) * 0.001
	local a = 1
	if tau > 0 then
		a = dtS / (tau + dtS)
	end
	if a < 0.02 then a = 0.02 end
	if a > 1.0 then a = 1.0 end

	cache.x = cache.x + dx * a
	cache.y = cache.y + dy * a
	cache.z = cache.z + dz * a

	return cache.x, cache.y, cache.z
end

local function sampleTerrainLayerAtWorldPos(x, z)
	if g_terrainNode == nil then return nil, nil, 0, 0 end
	if type(getTerrainNumOfLayers) ~= "function" then return nil, nil, 0, 0 end
	if type(getTerrainLayerAtWorldPos) ~= "function" then return nil, nil, 0, 0 end
	if type(getTerrainLayerName) ~= "function" then return nil, nil, 0, 0 end

	local numLayers = getTerrainNumOfLayers(g_terrainNode) or 0
	local bestId, bestW = nil, 0

	for layerId = 0, numLayers - 1 do
		local w = getTerrainLayerAtWorldPos(g_terrainNode, layerId, x, 0, z) or 0
		if w > bestW then
			bestW = w
			bestId = layerId
		end
	end

	if bestId ~= nil and bestW > 0 then
		return bestId, getTerrainLayerName(g_terrainNode, bestId), bestW, numLayers
	end

	return nil, nil, 0, numLayers
end

function WheelInspectorDebug:getControlledVehicle()
	if g_currentMission ~= nil and g_currentMission.hud ~= nil then
		return g_currentMission.hud.controlledVehicle
	end
	return nil
end

local function chooseWheelNode(wheel)
	if wheel == nil then return nil end

	local n = wheel.repr
	if isValidNode(n) then return n end

	n = wheel.driveNode
	if isValidNode(n) then return n end

	n = wheel.linkNode
	if isValidNode(n) then return n end

	if wheel.getFirstTireNode ~= nil then
		n = wheel:getFirstTireNode()
		if isValidNode(n) then return n end
	end

	n = wheel.node
	if isValidNode(n) then return n end

	return nil
end

-- ---------------------------------------------------------
-- wheel outline render
-- ---------------------------------------------------------
local function drawWheelOutline(wheelNode, radius, width, segments)
	if not isValidNode(wheelNode) then return end
	if radius == nil or radius <= 0 then return end

	segments = clamp(segments or 20, 6, 80)
	width = width or (radius * 0.35)

	local cx, cy, cz = getWorldTranslation(wheelNode)

	-- basis
	local axX, axY, axZ = localDirectionToWorld(wheelNode, 1, 0, 0) -- axle
	local upX, upY, upZ = localDirectionToWorld(wheelNode, 0, 1, 0)
	local fwX, fwY, fwZ = localDirectionToWorld(wheelNode, 0, 0, 1)

	axX, axY, axZ = norm(axX, axY, axZ)
	upX, upY, upZ = norm(upX, upY, upZ)
	fwX, fwY, fwZ = norm(fwX, fwY, fwZ)

	local halfW = width * 0.5
	local lx, ly, lz = cx - axX*halfW, cy - axY*halfW, cz - axZ*halfW
	local rx, ry, rz = cx + axX*halfW, cy + axY*halfW, cz + axZ*halfW

	local rr, gg, bb = 0.2, 1.0, 0.2
	local solid = false

	local prevLx, prevLy, prevLz = nil,nil,nil
	local prevRx, prevRy, prevRz = nil,nil,nil

	for i = 0, segments do
		local a = (i / segments) * (math.pi * 2)
		local ca = math.cos(a)
		local sa = math.sin(a)

		local offX = (upX * ca + fwX * sa) * radius
		local offY = (upY * ca + fwY * sa) * radius
		local offZ = (upZ * ca + fwZ * sa) * radius

		local pLx, pLy, pLz = lx + offX, ly + offY, lz + offZ
		local pRx, pRy, pRz = rx + offX, ry + offY, rz + offZ

		if prevLx ~= nil then
			drawLine(prevLx,prevLy,prevLz, pLx,pLy,pLz, rr,gg,bb, solid)
			drawLine(prevRx,prevRy,prevRz, pRx,pRy,pRz, rr,gg,bb, solid)
		end

		prevLx, prevLy, prevLz = pLx, pLy, pLz
		prevRx, prevRy, prevRz = pRx, pRy, pRz
	end

	local bridges = { {1,0}, {-1,0}, {0,1}, {0,-1} }
	for _, b in ipairs(bridges) do
		local cu, cf = b[1], b[2]
		local offX = (upX * cu + fwX * cf) * radius
		local offY = (upY * cu + fwY * cf) * radius
		local offZ = (upZ * cu + fwZ * cf) * radius
		drawLine(lx+offX, ly+offY, lz+offZ, rx+offX, ry+offY, rz+offZ, rr,gg,bb, solid)
	end

	drawLine(lx,ly,lz, rx,ry,rz, 1.0, 1.0, 0.1, solid)
end

local function buildWheelText(vehicle, wheel, idx, contactX, contactY, contactZ)
	local p = wheel ~= nil and wheel.physics or nil

	local lines = {}
	lines[#lines+1] = string.format("#%d", idx)

	lines[#lines+1] = string.format("r=%s  w=%s",
		fmtNum((p and p.radius) or wheel.radius, 3),
		fmtNum((p and p.width)  or wheel.width, 3)
	)

	lines[#lines+1] = string.format("mass=%s  fric=%s",
		fmtNum(p and p.mass, 3),
		fmtNum(p and p.frictionScale, 3)
	)

	lines[#lines+1] = string.format("shape=%s  soil=%s",
		tostring(p and p.wheelShapeCreated == true),
		tostring(p and p.hasSoilContact == true)
	)

	-- ---------------------------------------------------------
	-- Dirt values (0..1) from WASHABLE (правильно для визуалов)
	-- ---------------------------------------------------------
	local wheelDirty = nil
	local mudMeshDirty = nil

	if vehicle ~= nil and vehicle.spec_washable ~= nil and vehicle.getWashableNodeByCustomIndex ~= nil then
		-- 1) Dirt на колесе (scratches_dirt_snow_wetness) => customIndex = wheel
		local ndWheel = vehicle:getWashableNodeByCustomIndex(wheel)
		if ndWheel ~= nil then
			wheelDirty = ndWheel.dirtAmount
		end

		-- 2) Mud mesh (mudAmount) => customIndex = wheel.wheelMudMeshes (таблица нод)
		if wheel.wheelMudMeshes ~= nil then
			local ndMud = vehicle:getWashableNodeByCustomIndex(wheel.wheelMudMeshes)
			if ndMud ~= nil then
				mudMeshDirty = ndMud.dirtAmount
			end
		end
	end

	lines[#lines+1] = string.format("wheelDirty=%s  mudMesh=%s",
		fmtNum(wheelDirty, 3),
		fmtNum(mudMeshDirty, 3)
	)


	local surfaceName, terrainAttr = nil, nil
	if p ~= nil and p.getSurfaceSoundAttributes ~= nil then
		surfaceName, terrainAttr = p:getSurfaceSoundAttributes()
	end

	local densityType = p ~= nil and p.densityType or nil
	local fieldName = getFieldGroundTypeNameByValue(densityType)

	lines[#lines+1] = string.format("ground=%s  field=%s(%s)",
		tostring(surfaceName or "terrain"),
		fieldName,
		tostring(densityType)
	)

	local layerId, layerName, layerW = nil, nil, 0
	if contactX ~= nil and contactZ ~= nil then
		local cache = WheelInspectorDebug._terrainCache[wheel]
		local now = getTimeMs()
		if cache ~= nil and (now - (cache.t or 0)) < (WheelInspectorDebug.terrainSampleIntervalMs or 200) then
			layerId, layerName, layerW = cache.id, cache.name, cache.w
		else
			layerId, layerName, layerW = sampleTerrainLayerAtWorldPos(contactX, contactZ)
			WheelInspectorDebug._terrainCache[wheel] = { t = now, id = layerId, name = layerName, w = layerW }
		end
	end

	if layerId ~= nil then
		lines[#lines+1] = string.format("terrainLayer=%d (%s) w=%s",
			layerId, tostring(layerName), fmtNum(layerW, 3))
	else
		lines[#lines+1] = string.format("terrainLayer=none  terrainAttr=%s", tostring(terrainAttr))
	end

	if vehicle ~= nil then
		lines[#lines+1] = tostring(vehicle.typeName or vehicle.configFileName or "vehicle")
	end

	return table.concat(lines, "\n")
end

-- ---------------------------------------------------------
-- Console
-- ---------------------------------------------------------
function WheelInspectorDebug:cmdToggle(enableStr, segStr, rangeStr)
	if enableStr == nil or enableStr == "" then
		print("USAGE: gsWheelDbg [0|1] [segments] [range]")
		print("Example: gsWheelDbg 1 28 180")
		print(string.format("[WheelInspectorDebug] enabled=%s seg=%d range=%d",
			tostring(self.enabled), self.segments, self.maxRange))
		return
	end

	local b = toBool01(enableStr)
	if b == nil then
		print("gsWheelDbg: first arg must be 0/1")
		return
	end

	self.enabled = b
	self.segments = clamp(segStr or self.segments, 28, 80)
	self.maxRange = clamp(rangeStr or self.maxRange, 10, 5000)

	print(string.format("[WheelInspectorDebug] enabled=%s seg=%d range=%d",
		tostring(self.enabled), self.segments, self.maxRange))
end

local function tryRegister(self)
	if self._registered then return true end
	if type(addConsoleCommand) ~= "function" then return false end

	addConsoleCommand("gsWheelDbg", "Wheel overlay debug (current vehicle)", "cmdToggle", self)
	print("[WheelInspectorDebug] Registered via addConsoleCommand")

	self._registered = true
	return true
end

-- ---------------------------------------------------------
-- Draw hook
-- ---------------------------------------------------------
function WheelInspectorDebug:hookDrawOnce()
	if self._drawHooked then return end
	if FSBaseMission == nil or FSBaseMission.draw == nil then return end

	FSBaseMission.draw = Utils.overwrittenFunction(FSBaseMission.draw, function(mission, superFunc, ...)
		if superFunc ~= nil then
			superFunc(mission, ...)
		end

		if WheelInspectorDebug ~= nil and WheelInspectorDebug.enabled == true then
			WheelInspectorDebug:drawOverlay(mission.timeSinceLastFrame or 16)
		end
	end)

	self._drawHooked = true
	print("[WheelInspectorDebug] Hooked FSBaseMission.draw")
end

-- ---------------------------------------------------------
-- Actual drawing
-- ---------------------------------------------------------
function WheelInspectorDebug:drawOverlay(dt)
	if self.enabled ~= true then
		return
	end

	local veh = self:getControlledVehicle()
	local inVeh = (veh ~= nil)

	drawHudLine(string.format(
		"WheelDbg: ON  inVeh=%s  seg=%d  range=%dm",
		inVeh and "YES" or "NO",
		tonumber(self.segments or 0),
		tonumber(self.maxRange or 0)
	), 0.965)

	if not inVeh then
		drawHudLine("WheelDbg: enter a vehicle to see wheels", 0.945)
		return
	end

	if veh.spec_wheels == nil or veh.spec_wheels.wheels == nil then
		drawHudLine("WheelDbg: current vehicle has no spec_wheels!", 0.945)
		return
	end

	local camX,camY,camZ = getCamWorldPos()
	local maxD = self.maxRange or 120
	local maxD2 = maxD * maxD

	local wheels = veh.spec_wheels.wheels
	local found = #wheels
	local drawn = 0

	for i=1, found do
		local wheel = wheels[i]
		local n = chooseWheelNode(wheel)

		if n ~= nil then
			local wx,wy,wz = getWorldTranslation(n)
			if distSq(camX,camY,camZ, wx,wy,wz) <= maxD2 then
				local p = wheel.physics
				local radius = (p and p.radius) or wheel.radius or 0.6
				local width  = (p and p.width)  or wheel.width  or 0.45

				drawWheelOutline(n, radius, width, self.segments)

				local cx, cy, cz = getWheelContactWorldPos(n, p)

				cx, cy, cz = smoothWorldPosForWheel(wheel, cx, cy, cz, dt)

				local text = buildWheelText(veh, wheel, i, cx, cy, cz)
				drawWorldText(cx, cy + radius + 0.35, cz, text)


				drawn = drawn + 1
			end
		end
	end

	drawHudLine(string.format("WheelDbg: wheelsFound=%d wheelsDrawn=%d", found, drawn), 0.925)
end

-- ---------------------------------------------------------
-- Mod listener
-- ---------------------------------------------------------
function WheelInspectorDebug:loadMap()
	tryRegister(self)
	self:hookDrawOnce()
end

function WheelInspectorDebug:loadMapFinished()
	tryRegister(self)
	self:hookDrawOnce()
end

function WheelInspectorDebug:update(dt)
	if not self._registered then
		self._tryMs = (self._tryMs or 0) + (dt or 0)
		if self._tryMs > 1000 then
			self._tryMs = 0
			tryRegister(self)
		end
	end
	if not self._drawHooked then
		self._tryDrawMs = (self._tryDrawMs or 0) + (dt or 0)
		if self._tryDrawMs > 1000 then
			self._tryDrawMs = 0
			self:hookDrawOnce()
		end
	end
end
