--[[
    AttachImplementsManuallySettingsManager

    Helper class for ModSettings.

        @author: 	BayernGamers
        @date: 		07.06.2025
        @version:	1.0

        History:	v1.0 @07.06.2025 - initial implementation in FS25
                    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------

        License:    Terms:
                        Usage:
                            Feel free to use this work as-is as long as you adhere to the following terms:
                        Attribution:
                            You must give appropriate credit to the original author when using this work.
                        No Derivatives:
                            You may not alter, transform, or build upon this work in any way.
                        Usage:
                            The work may be used for personal and commercial purposes, provided it is not modified or adapted.
                        Additional Clause:
                            This script may not be converted, adapted, or incorporated into any other game versions or platforms except by GIANTS Software.
]]
source(Utils.getFilename("scripts/utils/LoggingUtil.lua", g_currentModDirectory))

local log = LoggingUtil.new(true, LoggingUtil.DEBUG_LEVELS.HIGH, "AttachImplementsManuallySettingsManager.lua")

AttachImplementsManuallySettingsManager = {}
AttachImplementsManuallySettingsManager.instance = nil
AttachImplementsManuallySettingsManager.MOD_DIRECTORY = g_currentModDirectory
AttachImplementsManuallySettingsManager.MOD_NAME = g_currentModName
AttachImplementsManuallySettingsManager.MOD_SETTINGS_DIR = g_currentModSettingsDirectory
AttachImplementsManuallySettingsManager.MOD_SETTINGS_FILE = Utils.getFilename("modSettings.xml", AttachImplementsManuallySettingsManager.MOD_SETTINGS_DIR)
AttachImplementsManuallySettingsManager.XML_SCHEMA = XMLSchema.new("attachImplementsManuallySettings")
AttachImplementsManuallySettingsManager.BASE_KEY = "modSettings"
AttachImplementsManuallySettingsManager.ENABLE_ATTACH_IMPLEMENTS_MANUALLY_KEY = AttachImplementsManuallySettingsManager.BASE_KEY .. ".enableAttachImplementsManually"
AttachImplementsManuallySettingsManager.ENABLE_AUTO_ATTACH_FRONTLOADER_TOOLS_KEY = AttachImplementsManuallySettingsManager.BASE_KEY .. ".enableAutoAttachFrontloaderTools"

local AttachImplementsManuallySettingsManager_mt = Class(AttachImplementsManuallySettingsManager)

function AttachImplementsManuallySettingsManager.getInstance()
    if AttachImplementsManuallySettingsManager.instance == nil then
        AttachImplementsManuallySettingsManager.instance = AttachImplementsManuallySettingsManager.new(AttachImplementsManuallySettingsManager.MOD_DIRECTORY, AttachImplementsManuallySettingsManager.MOD_SETTINGS_DIR)
    end

    return AttachImplementsManuallySettingsManager.instance
end

function AttachImplementsManuallySettingsManager.new(modDirectory, modSettingsDirectory)
    local self = setmetatable({}, AttachImplementsManuallySettingsManager_mt)
    local schema = AttachImplementsManuallySettingsManager.XML_SCHEMA
    createFolder(AttachImplementsManuallySettingsManager.MOD_SETTINGS_DIR)

    schema:register(XMLValueType.BOOL, AttachImplementsManuallySettingsManager.ENABLE_ATTACH_IMPLEMENTS_MANUALLY_KEY .. "#value", "Enable attaching implements manually", true)
    schema:register(XMLValueType.BOOL, AttachImplementsManuallySettingsManager.ENABLE_AUTO_ATTACH_FRONTLOADER_TOOLS_KEY .. "#value", "Enable auto attach frontloader tools", false)

    self.enableAttachImplementsManually = true
    self.enableAutoAttachFrontloaderTools = false
    self:load()
    
    return self
end

function AttachImplementsManuallySettingsManager:save()
    local xmlFile = XMLFile.loadIfExists("attachImplementsManuallySettings", AttachImplementsManuallySettingsManager.MOD_SETTINGS_FILE, AttachImplementsManuallySettingsManager.XML_SCHEMA)

    if xmlFile == nil then
        
        xmlFile = XMLFile.create("attachImplementsManuallySettings", AttachImplementsManuallySettingsManager.MOD_SETTINGS_FILE, "modSettings", AttachImplementsManuallySettingsManager.XML_SCHEMA)
    end

    xmlFile:setValue(AttachImplementsManuallySettingsManager.ENABLE_ATTACH_IMPLEMENTS_MANUALLY_KEY .. "#value", self.enableAttachImplementsManually)
    xmlFile:setValue(AttachImplementsManuallySettingsManager.ENABLE_AUTO_ATTACH_FRONTLOADER_TOOLS_KEY .. "#value", self.enableAutoAttachFrontloaderTools)

    xmlFile:save()
    xmlFile:delete()
end

function AttachImplementsManuallySettingsManager:load()
    local xmlFile = XMLFile.loadIfExists("attachImplementsManuallySettings", AttachImplementsManuallySettingsManager.MOD_SETTINGS_FILE, AttachImplementsManuallySettingsManager.XML_SCHEMA)

    if xmlFile ~= nil then
        self.enableAttachImplementsManually = xmlFile:getValue(AttachImplementsManuallySettingsManager.ENABLE_ATTACH_IMPLEMENTS_MANUALLY_KEY .. "#value", true)
        self.enableAutoAttachFrontloaderTools = xmlFile:getValue(AttachImplementsManuallySettingsManager.ENABLE_AUTO_ATTACH_FRONTLOADER_TOOLS_KEY .. "#value", false)
        xmlFile:delete()
    end
end

function AttachImplementsManuallySettingsManager:setEnableAttachImplementsManually(enabled)
    log:printDevInfo("Setting enableAttachImplementsManually to " .. tostring(enabled), LoggingUtil.DEBUG_LEVELS.HIGH)
    self.enableAttachImplementsManually = enabled
    self:save()
end

function AttachImplementsManuallySettingsManager:setEnableAutoAttachFrontloaderTools(enabled)
    log:printDevInfo("Setting enableAutoAttachFrontloaderTools to " .. tostring(enabled), LoggingUtil.DEBUG_LEVELS.HIGH)
    self.enableAutoAttachFrontloaderTools = enabled
    self:save()
end

function AttachImplementsManuallySettingsManager:getEnableAttachImplementsManually()
    return self.enableAttachImplementsManually
end

function AttachImplementsManuallySettingsManager:getEnableAutoAttachFrontloaderTools()
    return self.enableAutoAttachFrontloaderTools
end

function AttachImplementsManuallySettingsManager:getUserSettings()
    return self.enableAttachImplementsManually, self.enableAutoAttachFrontloaderTools
end
